import React from 'react';
import { Head, Link } from '@inertiajs/react';
import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { Card, CardContent, CardHeader, CardTitle } from '@/Components/ui/Card';
import { Button } from '@/Components/ui/Button';
import { Badge } from '@/Components/ui/Badge';
import { Clock, CheckCircle, XCircle, AlertCircle, FileText, TrendingUp } from 'lucide-react';

export default function Dashboard({ auth, stats, recent_revisions }) {
    const getStatusIcon = (status) => {
        switch (status) {
            case 'requested':
                return <Clock className="w-4 h-4 text-yellow-500" />;
            case 'in_progress':
                return <AlertCircle className="w-4 h-4 text-blue-500" />;
            case 'completed':
                return <CheckCircle className="w-4 h-4 text-green-500" />;
            case 'rejected':
                return <XCircle className="w-4 h-4 text-red-500" />;
            default:
                return <Clock className="w-4 h-4 text-gray-500" />;
        }
    };

    const getStatusBadge = (status) => {
        const statusConfig = {
            requested: { variant: 'secondary', text: 'Requested' },
            in_progress: { variant: 'default', text: 'In Progress' },
            completed: { variant: 'success', text: 'Completed' },
            rejected: { variant: 'destructive', text: 'Rejected' },
            escalated: { variant: 'warning', text: 'Escalated' },
        };

        const config = statusConfig[status] || { variant: 'secondary', text: status };
        return <Badge variant={config.variant}>{config.text}</Badge>;
    };

    const getTimeAgo = (dateString) => {
        const now = new Date();
        const date = new Date(dateString);
        const diffInHours = Math.floor((now - date) / (1000 * 60 * 60));
        
        if (diffInHours < 1) return 'Just now';
        if (diffInHours < 24) return `${diffInHours}h ago`;
        
        const diffInDays = Math.floor(diffInHours / 24);
        if (diffInDays < 7) return `${diffInDays}d ago`;
        
        return date.toLocaleDateString();
    };

    return (
        <AuthenticatedLayout
            user={auth.user}
            header={<h2 className="font-semibold text-xl text-gray-800 leading-tight">Revision Dashboard</h2>}
        >
            <Head title="Revision Dashboard" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {/* Header */}
                    <div className="mb-6">
                        <h1 className="text-3xl font-bold text-gray-900">Revision Dashboard</h1>
                        <p className="text-gray-600 mt-2">Overview of your revision work and performance</p>
                    </div>

                    {/* Statistics Grid */}
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                        <Card>
                            <CardContent className="p-6">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <FileText className="h-8 w-8 text-blue-600" />
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500">Total Revisions</p>
                                        <p className="text-2xl font-bold text-gray-900">{stats.total_revisions}</p>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>

                        <Card>
                            <CardContent className="p-6">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <Clock className="h-8 w-8 text-yellow-600" />
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500">Pending</p>
                                        <p className="text-2xl font-bold text-gray-900">{stats.pending_revisions}</p>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>

                        <Card>
                            <CardContent className="p-6">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <AlertCircle className="h-8 w-8 text-blue-600" />
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500">In Progress</p>
                                        <p className="text-2xl font-bold text-gray-900">{stats.in_progress_revisions}</p>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>

                        <Card>
                            <CardContent className="p-6">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <CheckCircle className="h-8 w-8 text-green-600" />
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500">Completed</p>
                                        <p className="text-2xl font-bold text-gray-900">{stats.completed_revisions}</p>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>
                    </div>

                    {/* Additional Stats Row */}
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
                        <Card>
                            <CardContent className="p-6">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <XCircle className="h-8 w-8 text-red-600" />
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500">Rejected</p>
                                        <p className="text-2xl font-bold text-gray-900">{stats.rejected_revisions}</p>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>

                        <Card>
                            <CardContent className="p-6">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <TrendingUp className="h-8 w-8 text-green-600" />
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500">Success Rate</p>
                                        <p className="text-2xl font-bold text-gray-900">
                                            {stats.total_revisions > 0 
                                                ? Math.round((stats.completed_revisions / stats.total_revisions) * 100)
                                                : 0}%
                                        </p>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>
                    </div>

                    {/* Recent Revisions */}
                    <Card>
                        <CardHeader>
                            <div className="flex items-center justify-between">
                                <CardTitle>Recent Revisions</CardTitle>
                                <Link
                                    href={route('writer.revisions.index')}
                                    className="inline-flex items-center px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2"
                                >
                                    View All Revisions
                                </Link>
                            </div>
                        </CardHeader>
                        <CardContent>
                            {recent_revisions.length === 0 ? (
                                <div className="text-center py-8">
                                    <FileText className="mx-auto h-12 w-12 text-gray-400" />
                                    <h3 className="mt-2 text-sm font-medium text-gray-900">No revisions yet</h3>
                                    <p className="mt-1 text-sm text-gray-500">
                                        You haven't been assigned any revisions yet.
                                    </p>
                                </div>
                            ) : (
                                <div className="space-y-4">
                                    {recent_revisions.map((revision) => (
                                        <div key={revision.id} className="flex items-center justify-between p-4 border border-gray-200 rounded-lg hover:bg-gray-50">
                                            <div className="flex items-center space-x-4">
                                                {getStatusIcon(revision.status)}
                                                <div>
                                                    <h4 className="text-sm font-medium text-gray-900">
                                                        {revision.order.title}
                                                    </h4>
                                                    <p className="text-sm text-gray-500">
                                                        Order #{revision.order.order_number} • {revision.order.client.name}
                                                    </p>
                                                    <div className="flex items-center space-x-2 mt-1">
                                                        {getStatusBadge(revision.status)}
                                                        <span className="text-xs text-gray-500">
                                                            {getTimeAgo(revision.updated_at)}
                                                        </span>
                                                    </div>
                                                </div>
                                            </div>
                                            <Link
                                                href={route('writer.revisions.show', revision.id)}
                                                className="inline-flex items-center px-3 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
                                            >
                                                View Details
                                            </Link>
                                        </div>
                                    ))}
                                </div>
                            )}
                        </CardContent>
                    </Card>

                    {/* Quick Actions */}
                    <Card className="mt-8">
                        <CardHeader>
                            <CardTitle>Quick Actions</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="flex flex-wrap gap-4">
                                <Link
                                    href={route('writer.revisions.index')}
                                    className="inline-flex items-center px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2"
                                >
                                    <FileText className="w-4 h-4 mr-2" />
                                    View All Revisions
                                </Link>
                                <Link
                                    href={route('writer.revisions.index', { status: 'requested' })}
                                    className="inline-flex items-center px-4 py-2 bg-yellow-600 text-white text-sm font-medium rounded-md hover:bg-yellow-700 focus:outline-none focus:ring-2 focus:ring-yellow-500 focus:ring-offset-2"
                                >
                                    <Clock className="w-4 h-4 mr-2" />
                                    Pending Revisions
                                </Link>
                                <Link
                                    href={route('writer.revisions.index', { status: 'in_progress' })}
                                    className="inline-flex items-center px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2"
                                >
                                    <AlertCircle className="w-4 h-4 mr-2" />
                                    In Progress
                                </Link>
                            </div>
                        </CardContent>
                    </Card>
                </div>
            </div>
        </AuthenticatedLayout>
    );
} 