import React, { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { Card, CardContent, CardHeader, CardTitle } from '@/Components/ui/Card';
import { Button } from '@/Components/ui/Button';
import { Badge } from '@/Components/ui/Badge';
import { Input } from '@/Components/ui/Input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/Components/ui/Select';
import { Search, Clock, CheckCircle, XCircle, AlertCircle } from 'lucide-react';

export default function Index({ auth, revisions, filters, statuses }) {
    const [search, setSearch] = useState(filters.search || '');
    const [status, setStatus] = useState(filters.status || '');

    const handleSearch = () => {
        router.get(route('writer.revisions.index'), { search, status }, {
            preserveState: true,
            replace: true,
        });
    };

    const handleStatusChange = (newStatus) => {
        setStatus(newStatus);
        router.get(route('writer.revisions.index'), { search, status: newStatus }, {
            preserveState: true,
            replace: true,
        });
    };

    const getStatusIcon = (status) => {
        switch (status) {
            case 'requested':
                return <Clock className="w-4 h-4 text-yellow-500" />;
            case 'in_progress':
                return <AlertCircle className="w-4 h-4 text-blue-500" />;
            case 'completed':
                return <CheckCircle className="w-4 h-4 text-green-500" />;
            case 'rejected':
                return <XCircle className="w-4 h-4 text-red-500" />;
            default:
                return <Clock className="w-4 h-4 text-gray-500" />;
        }
    };

    const getStatusBadge = (status) => {
        const statusConfig = {
            requested: { variant: 'secondary', text: 'Requested' },
            in_progress: { variant: 'default', text: 'In Progress' },
            completed: { variant: 'success', text: 'Completed' },
            rejected: { variant: 'destructive', text: 'Rejected' },
            escalated: { variant: 'warning', text: 'Escalated' },
        };

        const config = statusConfig[status] || { variant: 'secondary', text: status };
        return <Badge variant={config.variant}>{config.text}</Badge>;
    };

    return (
        <AuthenticatedLayout
            user={auth.user}
            header={<h2 className="font-semibold text-xl text-gray-800 leading-tight">My Revisions</h2>}
        >
            <Head title="My Revisions" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {/* Header */}
                    <div className="mb-6">
                        <h1 className="text-3xl font-bold text-gray-900">Revision Management</h1>
                        <p className="text-gray-600 mt-2">Manage and track all your assigned revisions</p>
                    </div>

                    {/* Filters */}
                    <Card className="mb-6">
                        <CardContent className="p-6">
                            <div className="flex flex-col sm:flex-row gap-4">
                                <div className="flex-1">
                                    <Input
                                        placeholder="Search by order title or number..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
                                        className="w-full"
                                    />
                                </div>
                                <div className="w-full sm:w-48">
                                    <Select value={status} onValueChange={handleStatusChange}>
                                        <SelectTrigger>
                                            <SelectValue placeholder="All Statuses" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="">All Statuses</SelectItem>
                                            {Object.entries(statuses).map(([key, value]) => (
                                                <SelectItem key={key} value={key}>
                                                    {value}
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                </div>
                                <Button onClick={handleSearch} className="w-full sm:w-auto">
                                    <Search className="w-4 h-4 mr-2" />
                                    Search
                                </Button>
                            </div>
                        </CardContent>
                    </Card>

                    {/* Revisions List */}
                    <div className="space-y-4">
                        {revisions.data.length === 0 ? (
                            <Card>
                                <CardContent className="p-8 text-center">
                                    <div className="text-gray-500">
                                        <Clock className="w-16 h-16 mx-auto mb-4 text-gray-300" />
                                        <h3 className="text-lg font-medium mb-2">No revisions found</h3>
                                        <p className="text-sm">
                                            {search || status ? 'Try adjusting your search criteria.' : 'You don\'t have any revisions assigned yet.'}
                                        </p>
                                    </div>
                                </CardContent>
                            </Card>
                        ) : (
                            revisions.data.map((revision) => (
                                <Card key={revision.id} className="hover:shadow-md transition-shadow">
                                    <CardContent className="p-6">
                                        <div className="flex items-start justify-between">
                                            <div className="flex-1">
                                                <div className="flex items-center gap-3 mb-3">
                                                    {getStatusIcon(revision.status)}
                                                    <div>
                                                        <h3 className="text-lg font-semibold text-gray-900">
                                                            {revision.order.title}
                                                        </h3>
                                                        <p className="text-sm text-gray-600">
                                                            Order #{revision.order.order_number}
                                                        </p>
                                                    </div>
                                                </div>

                                                <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                                                    <div>
                                                        <span className="text-sm font-medium text-gray-500">Revision Type</span>
                                                        <p className="text-sm text-gray-900 capitalize">
                                                            {revision.revision_type.replace('_', ' ')}
                                                        </p>
                                                    </div>
                                                    <div>
                                                        <span className="text-sm font-medium text-gray-500">Due Date</span>
                                                        <p className="text-sm text-gray-900">
                                                            {revision.due_date ? new Date(revision.due_date).toLocaleDateString() : 'Not set'}
                                                        </p>
                                                    </div>
                                                    <div>
                                                        <span className="text-sm font-medium text-gray-500">Status</span>
                                                        <div className="mt-1">
                                                            {getStatusBadge(revision.status)}
                                                        </div>
                                                    </div>
                                                </div>

                                                {revision.reason && (
                                                    <div className="mb-4">
                                                        <span className="text-sm font-medium text-gray-500">Reason</span>
                                                        <p className="text-sm text-gray-900 mt-1">{revision.reason}</p>
                                                    </div>
                                                )}

                                                {revision.instructions && (
                                                    <div className="mb-4">
                                                        <span className="text-sm font-medium text-gray-500">Instructions</span>
                                                        <p className="text-sm text-gray-900 mt-1">{revision.instructions}</p>
                                                    </div>
                                                )}
                                            </div>

                                            <div className="flex flex-col gap-2 ml-4">
                                                <Link
                                                    href={route('writer.revisions.show', revision.id)}
                                                    className="inline-flex items-center justify-center px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2"
                                                >
                                                    View Details
                                                </Link>

                                                {revision.status === 'requested' && (
                                                    <Button
                                                        onClick={() => router.post(route('writer.revisions.start-working', revision.id))}
                                                        variant="default"
                                                        size="sm"
                                                    >
                                                        Start Working
                                                    </Button>
                                                )}

                                                {revision.status === 'in_progress' && (
                                                    <Button
                                                        onClick={() => router.post(route('writer.revisions.mark-complete', revision.id))}
                                                        variant="success"
                                                        size="sm"
                                                    >
                                                        Mark Complete
                                                    </Button>
                                                )}
                                            </div>
                                        </div>
                                    </CardContent>
                                </Card>
                            ))
                        )}
                    </div>

                    {/* Pagination */}
                    {revisions.links && revisions.links.length > 3 && (
                        <div className="mt-6">
                            <nav className="flex justify-center">
                                <div className="flex space-x-1">
                                    {revisions.links.map((link, key) => (
                                        <Link
                                            key={key}
                                            href={link.url}
                                            className={`px-3 py-2 text-sm font-medium rounded-md ${
                                                link.url === null
                                                    ? 'text-gray-400 cursor-not-allowed'
                                                    : link.active
                                                    ? 'bg-blue-600 text-white'
                                                    : 'text-gray-700 bg-white border border-gray-300 hover:bg-gray-50'
                                            }`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </nav>
                        </div>
                    )}
                </div>
            </div>
        </AuthenticatedLayout>
    );
} 