import React, { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { Card, CardContent, CardHeader, CardTitle } from '@/Components/ui/Card';
import { Button } from '@/Components/ui/Button';
import { Badge } from '@/Components/ui/Badge';
import { Textarea } from '@/Components/ui/Textarea';
import { Clock, CheckCircle, XCircle, AlertCircle, FileText, User, Calendar, MessageSquare } from 'lucide-react';

export default function Show({ auth, revision, order, conversation }) {
    const [completionNotes, setCompletionNotes] = useState('');
    const [isSubmitting, setIsSubmitting] = useState(false);

    const getStatusIcon = (status) => {
        switch (status) {
            case 'requested':
                return <Clock className="w-5 h-5 text-yellow-500" />;
            case 'in_progress':
                return <AlertCircle className="w-5 h-5 text-blue-500" />;
            case 'completed':
                return <CheckCircle className="w-5 h-5 text-green-500" />;
            case 'rejected':
                return <XCircle className="w-5 h-5 text-red-500" />;
            default:
                return <Clock className="w-5 h-5 text-gray-500" />;
        }
    };

    const getStatusBadge = (status) => {
        const statusConfig = {
            requested: { variant: 'secondary', text: 'Requested' },
            in_progress: { variant: 'default', text: 'In Progress' },
            completed: { variant: 'success', text: 'Completed' },
            rejected: { variant: 'destructive', text: 'Rejected' },
            escalated: { variant: 'warning', text: 'Escalated' },
        };

        const config = statusConfig[status] || { variant: 'secondary', text: status };
        return <Badge variant={config.variant}>{config.text}</Badge>;
    };

    const getRevisionTypeLabel = (type) => {
        const typeLabels = {
            quality: 'Content Quality Issues',
            requirements: 'Requirements Not Met',
            formatting: 'Formatting Issues',
            plagiarism: 'Plagiarism Concerns',
            deadline: 'Deadline Issues',
            communication: 'Communication Issues',
            other: 'Other Issues',
        };

        return typeLabels[type] || type;
    };

    const handleStartWorking = () => {
        setIsSubmitting(true);
        router.post(route('writer.revisions.start-working', revision.id), {}, {
            onFinish: () => setIsSubmitting(false),
        });
    };

    const handleMarkComplete = () => {
        if (!completionNotes.trim()) {
            alert('Please provide completion notes before marking as complete.');
            return;
        }

        setIsSubmitting(true);
        router.post(route('writer.revisions.mark-complete', revision.id), {
            completion_notes: completionNotes,
        }, {
            onFinish: () => setIsSubmitting(false),
        });
    };

    const formatDate = (dateString) => {
        if (!dateString) return 'Not set';
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit',
        });
    };

    return (
        <AuthenticatedLayout
            user={auth.user}
            header={<h2 className="font-semibold text-xl text-gray-800 leading-tight">Revision Details</h2>}
        >
            <Head title={`Revision - ${order.title}`} />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {/* Header */}
                    <div className="mb-6">
                        <div className="flex items-center justify-between">
                            <div>
                                <h1 className="text-3xl font-bold text-gray-900">Revision Details</h1>
                                <p className="text-gray-600 mt-2">
                                    Order #{order.order_number} - {order.title}
                                </p>
                            </div>
                            <div className="flex items-center space-x-3">
                                <Link
                                    href={route('writer.revisions.index')}
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
                                >
                                    <FileText className="w-4 h-4 mr-2" />
                                    Back to Revisions
                                </Link>
                                <Link
                                    href={route('writer.revisions.dashboard')}
                                    className="inline-flex items-center px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-md hover:bg-blue-700"
                                >
                                    Dashboard
                                </Link>
                            </div>
                        </div>
                    </div>

                    <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                        {/* Main Content */}
                        <div className="lg:col-span-2 space-y-6">
                            {/* Revision Overview */}
                            <Card>
                                <CardHeader>
                                    <div className="flex items-center space-x-3">
                                        {getStatusIcon(revision.status)}
                                        <div>
                                            <CardTitle>Revision Overview</CardTitle>
                                            <p className="text-sm text-gray-600">
                                                Status: {getStatusBadge(revision.status)}
                                            </p>
                                        </div>
                                    </div>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <span className="text-sm font-medium text-gray-500">Revision Type</span>
                                            <p className="text-sm text-gray-900 mt-1">
                                                {getRevisionTypeLabel(revision.revision_type)}
                                            </p>
                                        </div>
                                        <div>
                                            <span className="text-sm font-medium text-gray-500">Revision Number</span>
                                            <p className="text-sm text-gray-900 mt-1">
                                                #{revision.revision_number}
                                            </p>
                                        </div>
                                        <div>
                                            <span className="text-sm font-medium text-gray-500">Due Date</span>
                                            <p className="text-sm text-gray-900 mt-1">
                                                {formatDate(revision.due_date)}
                                            </p>
                                        </div>
                                        <div>
                                            <span className="text-sm font-medium text-gray-500">Created</span>
                                            <p className="text-sm text-gray-900 mt-1">
                                                {formatDate(revision.created_at)}
                                            </p>
                                        </div>
                                    </div>

                                    {revision.reason && (
                                        <div>
                                            <span className="text-sm font-medium text-gray-500">Reason for Revision</span>
                                            <p className="text-sm text-gray-900 mt-1">{revision.reason}</p>
                                        </div>
                                    )}

                                    {revision.instructions && (
                                        <div>
                                            <span className="text-sm font-medium text-gray-500">Specific Instructions</span>
                                            <p className="text-sm text-gray-900 mt-1">{revision.instructions}</p>
                                        </div>
                                    )}

                                    {revision.client_notes && (
                                        <div>
                                            <span className="text-sm font-medium text-gray-500">Client Notes</span>
                                            <p className="text-sm text-gray-900 mt-1">{revision.client_notes}</p>
                                        </div>
                                    )}

                                    {revision.admin_notes && (
                                        <div>
                                            <span className="text-sm font-medium text-gray-500">Admin Notes</span>
                                            <p className="text-sm text-gray-900 mt-1">{revision.admin_notes}</p>
                                        </div>
                                    )}
                                </CardContent>
                            </Card>

                            {/* Order Information */}
                            <Card>
                                <CardHeader>
                                    <CardTitle>Order Information</CardTitle>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <span className="text-sm font-medium text-gray-500">Order Title</span>
                                            <p className="text-sm text-gray-900 mt-1">{order.title}</p>
                                        </div>
                                        <div>
                                            <span className="text-sm font-medium text-gray-500">Subject</span>
                                            <p className="text-sm text-gray-900 mt-1">{order.subject}</p>
                                        </div>
                                        <div>
                                            <span className="text-sm font-medium text-gray-500">Academic Level</span>
                                            <p className="text-sm text-gray-900 mt-1">{order.academic_level}</p>
                                        </div>
                                        <div>
                                            <span className="text-sm font-medium text-gray-500">Order Status</span>
                                            <p className="text-sm text-gray-900 mt-1 capitalize">
                                                {order.status.replace('_', ' ')}
                                            </p>
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>

                            {/* Action Buttons */}
                            {revision.status === 'requested' && (
                                <Card>
                                    <CardHeader>
                                        <CardTitle>Start Working</CardTitle>
                                    </CardHeader>
                                    <CardContent>
                                        <p className="text-sm text-gray-600 mb-4">
                                            Click the button below to start working on this revision. This will change the status to "In Progress".
                                        </p>
                                        <Button
                                            onClick={handleStartWorking}
                                            disabled={isSubmitting}
                                            className="w-full sm:w-auto"
                                        >
                                            <AlertCircle className="w-4 h-4 mr-2" />
                                            {isSubmitting ? 'Starting...' : 'Start Working'}
                                        </Button>
                                    </CardContent>
                                </Card>
                            )}

                            {revision.status === 'in_progress' && (
                                <Card>
                                    <CardHeader>
                                        <CardTitle>Mark as Complete</CardTitle>
                                    </CardHeader>
                                    <CardContent className="space-y-4">
                                        <div>
                                            <label htmlFor="completion_notes" className="block text-sm font-medium text-gray-700 mb-2">
                                                Completion Notes
                                            </label>
                                            <Textarea
                                                id="completion_notes"
                                                value={completionNotes}
                                                onChange={(e) => setCompletionNotes(e.target.value)}
                                                placeholder="Describe what was completed or any notes about the revision..."
                                                rows={4}
                                                className="w-full"
                                            />
                                        </div>
                                        <Button
                                            onClick={handleMarkComplete}
                                            disabled={isSubmitting || !completionNotes.trim()}
                                            variant="success"
                                            className="w-full sm:w-auto"
                                        >
                                            <CheckCircle className="w-4 h-4 mr-2" />
                                            {isSubmitting ? 'Marking Complete...' : 'Mark as Complete'}
                                        </Button>
                                    </CardContent>
                                </Card>
                            )}

                            {/* Timeline */}
                            <Card>
                                <CardHeader>
                                    <CardTitle>Revision Timeline</CardTitle>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-4">
                                        <div className="flex items-start space-x-3">
                                            <div className="flex-shrink-0">
                                                <div className="w-3 h-3 bg-blue-500 rounded-full"></div>
                                            </div>
                                            <div>
                                                <p className="text-sm font-medium text-gray-900">Revision Requested</p>
                                                <p className="text-sm text-gray-500">{formatDate(revision.created_at)}</p>
                                            </div>
                                        </div>

                                        {revision.started_at && (
                                            <div className="flex items-start space-x-3">
                                                <div className="flex-shrink-0">
                                                    <div className="w-3 h-3 bg-blue-500 rounded-full"></div>
                                                </div>
                                                <div>
                                                    <p className="text-sm font-medium text-gray-900">Work Started</p>
                                                    <p className="text-sm text-gray-500">{formatDate(revision.started_at)}</p>
                                                </div>
                                            </div>
                                        )}

                                        {revision.completed_at && (
                                            <div className="flex items-start space-x-3">
                                                <div className="flex-shrink-0">
                                                    <div className="w-3 h-3 bg-green-500 rounded-full"></div>
                                                </div>
                                                <div>
                                                    <p className="text-sm font-medium text-gray-900">Completed</p>
                                                    <p className="text-sm text-gray-500">{formatDate(revision.completed_at)}</p>
                                                </div>
                                            </div>
                                        )}

                                        {revision.due_date && (
                                            <div className="flex items-start space-x-3">
                                                <div className="flex-shrink-0">
                                                    <div className="w-3 h-3 bg-yellow-500 rounded-full"></div>
                                                </div>
                                                <div>
                                                    <p className="text-sm font-medium text-gray-900">Due Date</p>
                                                    <p className="text-sm text-gray-500">{formatDate(revision.due_date)}</p>
                                                </div>
                                            </div>
                                        )}
                                    </div>
                                </CardContent>
                            </Card>
                        </div>

                        {/* Sidebar */}
                        <div className="space-y-6">
                            {/* Client Information */}
                            <Card>
                                <CardHeader>
                                    <CardTitle>Client Information</CardTitle>
                                </CardHeader>
                                <CardContent>
                                    <div className="flex items-center space-x-3">
                                        <User className="w-8 h-8 text-gray-400" />
                                        <div>
                                            <p className="text-sm font-medium text-gray-900">{order.client.name}</p>
                                            <p className="text-sm text-gray-500">{order.client.email}</p>
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>

                            {/* Conversation */}
                            {conversation && (
                                <Card>
                                    <CardHeader>
                                        <CardTitle>Conversation</CardTitle>
                                    </CardHeader>
                                    <CardContent>
                                        <div className="flex items-center space-x-3 mb-3">
                                            <MessageSquare className="w-5 h-5 text-gray-400" />
                                            <span className="text-sm text-gray-600">
                                                {conversation.messages?.length || 0} messages
                                            </span>
                                        </div>
                                        <Link
                                            href={`/conversations/${conversation.id}`}
                                            className="inline-flex items-center px-3 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50 w-full justify-center"
                                        >
                                            View Conversation
                                        </Link>
                                    </CardContent>
                                </Card>
                            )}

                            {/* Quick Actions */}
                            <Card>
                                <CardHeader>
                                    <CardTitle>Quick Actions</CardTitle>
                                </CardHeader>
                                <CardContent className="space-y-3">
                                    <Link
                                        href={route('writer.revisions.index')}
                                        className="inline-flex items-center justify-center w-full px-3 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
                                    >
                                        <FileText className="w-4 h-4 mr-2" />
                                        All Revisions
                                    </Link>
                                    <Link
                                        href={route('writer.revisions.dashboard')}
                                        className="inline-flex items-center justify-center w-full px-3 py-2 bg-blue-600 text-white text-sm font-medium rounded-md hover:bg-blue-700"
                                    >
                                        <Calendar className="w-4 h-4 mr-2" />
                                        Dashboard
                                    </Link>
                                </CardContent>
                            </Card>
                        </div>
                    </div>
                </div>
            </div>
        </AuthenticatedLayout>
    );
} 