import { useState, useEffect, useRef } from "react";
import { Head, useForm } from "@inertiajs/react";
import WriterLayout from "@/Layouts/WriterLayout";
import axios from "axios";
import {
    Clock,
    AlertTriangle,
    CheckCircle,
    ArrowRight,
    Send,
    HelpCircle,
} from "lucide-react";

export default function Take({
    questions,
    totalQuestions,
    timeRemaining,
    testStartedAt,
    sessionToken,
}) {
    const [selectedAnswers, setSelectedAnswers] = useState({});
    const [currentQuestion, setCurrentQuestion] = useState(0);
    const [remainingTime, setRemainingTime] = useState(timeRemaining);
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [showWarning, setShowWarning] = useState(false);
    const [warningLevel, setWarningLevel] = useState(null); // null, 'low', 'medium', 'high'
    const [isAutoSubmitting, setIsAutoSubmitting] = useState(false);
    const timerRef = useRef(null);
    const pingRef = useRef(null);
    const timeCheckRef = useRef(null);
    const autoSubmitTimeoutRef = useRef(null);

    const { data, setData, post, processing, errors } = useForm({
        answers: {},
    });

    // Format time in seconds to HH:MM:SS
    const formatTime = (seconds) => {
        const hours = Math.floor(seconds / 3600);
        const minutes = Math.floor((seconds % 3600) / 60);
        const secs = seconds % 60;
        return `${hours.toString().padStart(2, "0")}:${minutes
            .toString()
            .padStart(2, "0")}:${secs.toString().padStart(2, "0")}`;
    };

    // Handle answer selection
    const handleAnswerSelect = (questionId, answer) => {
        const newAnswers = { ...selectedAnswers, [questionId]: answer };
        setSelectedAnswers(newAnswers);
        setData("answers", newAnswers);
    };

    // Navigate to next question
    const nextQuestion = () => {
        if (currentQuestion < questions.length - 1) {
            setCurrentQuestion(currentQuestion + 1);
        }
    };

    // Navigate to previous question
    const prevQuestion = () => {
        if (currentQuestion > 0) {
            setCurrentQuestion(currentQuestion - 1);
        }
    };

    // Jump to a specific question
    const jumpToQuestion = (index) => {
        if (index >= 0 && index < questions.length) {
            setCurrentQuestion(index);
        }
    };

    // Check if a question has been answered
    const isQuestionAnswered = (questionId) => {
        return selectedAnswers[questionId] !== undefined;
    };

    // Count answered questions
    const countAnsweredQuestions = () => {
        return Object.keys(selectedAnswers).length;
    };

    // Submit the test
    const submitTest = () => {
        if (countAnsweredQuestions() === 0) {
            if (
                !confirm(
                    "You haven't answered any questions. Are you sure you want to submit?"
                )
            ) {
                return;
            }
        } else if (countAnsweredQuestions() < totalQuestions) {
            if (
                !confirm(
                    `You've only answered ${countAnsweredQuestions()} out of ${totalQuestions} questions. Are you sure you want to submit?`
                )
            ) {
                return;
            }
        }

        setIsSubmitting(true);
        post(route("writer.test.submit"), {
            onSuccess: () => {
                // Submission is handled by the controller
            },
            onError: () => {
                setIsSubmitting(false);
            },
        });
    };

    // Record activity to prevent session timeout
    const recordActivity = async () => {
        try {
            await axios.post(route("writer.test.record-activity"), {
                sessionToken: sessionToken,
            });
        } catch (error) {
            console.error("Error recording activity:", error);
        }
    };

    // Check remaining time from server
    const checkRemainingTime = async () => {
        try {
            const response = await axios.get(route("writer.test.check-time"), {
                params: { sessionToken },
            });

            // If server indicates the test is expired, handle it
            if (response.data.isExpired) {
                handleAutoSubmit();
                return;
            }

            if (response.data.remainingTime !== undefined) {
                setRemainingTime(response.data.remainingTime);
            }
        } catch (error) {
            console.error("Error checking time:", error);
            // If we get a redirect instruction, follow it
            if (error.response?.data?.redirect) {
                window.location.href = error.response.data.redirect;
            }
        }
    };

    // Auto-submit the test when time expires
    const handleAutoSubmit = async () => {
        // Prevent multiple auto-submissions
        if (isAutoSubmitting) return;

        setIsAutoSubmitting(true);

        try {
            const token = document
                .querySelector('meta[name="csrf-token"]')
                ?.getAttribute("content");

            const response = await axios.post(
                route("writer.test.auto-submit"),
                {
                    sessionToken,
                    answers: selectedAnswers,
                },
                {
                    headers: {
                        "X-CSRF-TOKEN": token,
                        "Content-Type": "application/json",
                    },
                }
            );

            if (response.data.redirect) {
                window.location.href = response.data.redirect;
            }
        } catch (error) {
            console.error("Auto-submission error:", error);
            // Even if auto-submission fails, redirect to results
            window.location.href = route("writer.test.results");
        }
    };

    // Update warning level based on remaining time
    const updateWarningLevel = (seconds) => {
        if (seconds <= 60) {
            // 1 minute or less
            setWarningLevel("high");
            setShowWarning(true);
        } else if (seconds <= 300) {
            // 5 minutes or less
            setWarningLevel("medium");
            setShowWarning(true);
        } else if (seconds <= 600) {
            // 10 minutes or less
            setWarningLevel("low");
            setShowWarning(true);
        } else {
            setWarningLevel(null);
            setShowWarning(false);
        }
    };

    // Initialize timers and cleanup on unmount
    useEffect(() => {
        // Initial time check
        checkRemainingTime();

        // Set up timer to update countdown
        timerRef.current = setInterval(() => {
            setRemainingTime((prevTime) => {
                if (prevTime <= 1) {
                    // Time's up, trigger auto-submission
                    clearInterval(timerRef.current);
                    handleAutoSubmit();
                    return 0;
                }

                // Update warning level
                updateWarningLevel(prevTime - 1);

                return prevTime - 1;
            });
        }, 1000);

        // Set up periodic time check with server
        timeCheckRef.current = setInterval(checkRemainingTime, 30000); // Every 30 seconds

        // Set up activity recording
        pingRef.current = setInterval(recordActivity, 60000); // Every minute

        // Set up auto-submit timeout as a backup
        // This ensures auto-submit happens even if the interval is delayed
        if (timeRemaining) {
            autoSubmitTimeoutRef.current = setTimeout(() => {
                handleAutoSubmit();
            }, timeRemaining * 1000 + 1000); // Add 1 second buffer
        }

        // Cleanup on unmount
        return () => {
            clearInterval(timerRef.current);
            clearInterval(timeCheckRef.current);
            clearInterval(pingRef.current);
            clearTimeout(autoSubmitTimeoutRef.current);
        };
    }, []);

    // Set up beforeunload event to warn user before leaving page
    useEffect(() => {
        const handleBeforeUnload = (e) => {
            if (remainingTime > 0 && !isSubmitting && !isAutoSubmitting) {
                const message =
                    "You are in the middle of your test. If you leave, your progress may not be saved.";
                e.returnValue = message;
                return message;
            }
        };

        window.addEventListener("beforeunload", handleBeforeUnload);

        return () => {
            window.removeEventListener("beforeunload", handleBeforeUnload);
        };
    }, [remainingTime, isSubmitting, isAutoSubmitting]);

    return (
        <WriterLayout>
            <Head title="Writer Qualification Test" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6 bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
                            <div className="flex justify-between items-center mb-6">
                                <h1 className="text-2xl font-bold text-gray-800 dark:text-gray-100">
                                    Writer Qualification Test
                                </h1>
                                <div
                                    className={`flex items-center px-4 py-2 rounded-lg ${
                                        warningLevel === "high"
                                            ? "bg-red-50 dark:bg-red-900/30 text-red-700 dark:text-red-300 animate-pulse"
                                            : warningLevel === "medium"
                                            ? "bg-orange-50 dark:bg-orange-900/30 text-orange-700 dark:text-orange-300"
                                            : "bg-blue-50 dark:bg-blue-900/30 text-blue-700 dark:text-blue-300"
                                    }`}
                                >
                                    <Clock
                                        className={`h-5 w-5 mr-2 ${
                                            warningLevel === "high"
                                                ? "text-red-500 dark:text-red-400"
                                                : warningLevel === "medium"
                                                ? "text-orange-500 dark:text-orange-400"
                                                : "text-blue-500 dark:text-blue-400"
                                        }`}
                                    />
                                    <span className="font-mono">
                                        {formatTime(remainingTime)}
                                    </span>
                                </div>
                            </div>

                            {/* Time warning */}
                            {showWarning && (
                                <div
                                    className={`mb-6 p-4 border-l-4 ${
                                        warningLevel === "high"
                                            ? "bg-red-50 dark:bg-red-900/20 border-red-500 text-red-700 dark:text-red-300"
                                            : warningLevel === "medium"
                                            ? "bg-orange-50 dark:bg-orange-900/20 border-orange-500 text-orange-700 dark:text-orange-300"
                                            : "bg-yellow-50 dark:bg-yellow-900/20 border-yellow-400 text-yellow-700 dark:text-yellow-300"
                                    }`}
                                >
                                    <div className="flex">
                                        <div className="flex-shrink-0">
                                            <AlertTriangle
                                                className={`h-5 w-5 ${
                                                    warningLevel === "high"
                                                        ? "text-red-500"
                                                        : warningLevel ===
                                                          "medium"
                                                        ? "text-orange-500"
                                                        : "text-yellow-500"
                                                }`}
                                                aria-hidden="true"
                                            />
                                        </div>
                                        <div className="ml-3">
                                            <p className="text-sm font-medium">
                                                {warningLevel === "high"
                                                    ? "Time is almost up! Your test will be automatically submitted in less than a minute."
                                                    : warningLevel === "medium"
                                                    ? "Warning: Less than 5 minutes remaining. Please finish and submit your test soon."
                                                    : "Notice: Less than 10 minutes remaining. Consider finalizing your answers."}
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            )}

                            {/* Progress indicator */}
                            <div className="mb-6">
                                <div className="flex justify-between items-center mb-2">
                                    <span className="text-sm text-gray-600 dark:text-gray-400">
                                        Question {currentQuestion + 1} of{" "}
                                        {totalQuestions}
                                    </span>
                                    <span className="text-sm text-gray-600 dark:text-gray-400">
                                        {countAnsweredQuestions()} of{" "}
                                        {totalQuestions} answered
                                    </span>
                                </div>
                                <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-2.5">
                                    <div
                                        className="bg-blue-600 h-2.5 rounded-full"
                                        style={{
                                            width: `${
                                                (countAnsweredQuestions() /
                                                    totalQuestions) *
                                                100
                                            }%`,
                                        }}
                                    ></div>
                                </div>
                            </div>

                            {/* Question navigation */}
                            <div className="mb-6 flex flex-wrap gap-2">
                                {questions.map((question, index) => (
                                    <button
                                        key={question.id}
                                        onClick={() => jumpToQuestion(index)}
                                        className={`w-10 h-10 rounded-full flex items-center justify-center text-sm font-medium ${
                                            currentQuestion === index
                                                ? "bg-blue-600 text-white"
                                                : isQuestionAnswered(
                                                      question.id
                                                  )
                                                ? "bg-green-100 dark:bg-green-900/30 text-green-800 dark:text-green-300 border border-green-300 dark:border-green-700"
                                                : "bg-gray-100 dark:bg-gray-700 text-gray-800 dark:text-gray-300 border border-gray-300 dark:border-gray-600"
                                        }`}
                                    >
                                        {index + 1}
                                    </button>
                                ))}
                            </div>

                            {/* Current question */}
                            {questions.length > 0 && (
                                <div className="mb-8">
                                    <div className="mb-4">
                                        <span className="inline-flex items-center px-2.5 py-0.5 rounded-md text-sm font-medium bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-300">
                                            {
                                                questions[currentQuestion]
                                                    .category
                                            }
                                        </span>
                                        <h2 className="mt-2 text-xl font-semibold text-gray-800 dark:text-gray-100">
                                            {
                                                questions[currentQuestion]
                                                    .question
                                            }
                                        </h2>
                                    </div>

                                    <div className="space-y-3">
                                        {Object.entries(
                                            questions[currentQuestion].options
                                        ).map(([key, value]) => (
                                            <div
                                                key={key}
                                                className={`p-4 border rounded-lg cursor-pointer transition-colors ${
                                                    selectedAnswers[
                                                        questions[
                                                            currentQuestion
                                                        ].id
                                                    ] === key
                                                        ? "bg-blue-50 dark:bg-blue-900/20 border-blue-300 dark:border-blue-700"
                                                        : "bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700 hover:bg-gray-50 dark:hover:bg-gray-700/50"
                                                }`}
                                                onClick={() =>
                                                    handleAnswerSelect(
                                                        questions[
                                                            currentQuestion
                                                        ].id,
                                                        key
                                                    )
                                                }
                                            >
                                                <div className="flex items-start">
                                                    <div className="flex-shrink-0 mt-0.5">
                                                        <div
                                                            className={`w-5 h-5 rounded-full border flex items-center justify-center ${
                                                                selectedAnswers[
                                                                    questions[
                                                                        currentQuestion
                                                                    ].id
                                                                ] === key
                                                                    ? "border-blue-500 bg-blue-500 dark:border-blue-400 dark:bg-blue-400"
                                                                    : "border-gray-300 dark:border-gray-600"
                                                            }`}
                                                        >
                                                            {selectedAnswers[
                                                                questions[
                                                                    currentQuestion
                                                                ].id
                                                            ] === key && (
                                                                <CheckCircle className="w-4 h-4 text-white" />
                                                            )}
                                                        </div>
                                                    </div>
                                                    <div className="ml-3">
                                                        <span
                                                            className={`font-medium ${
                                                                selectedAnswers[
                                                                    questions[
                                                                        currentQuestion
                                                                    ].id
                                                                ] === key
                                                                    ? "text-blue-700 dark:text-blue-300"
                                                                    : "text-gray-700 dark:text-gray-300"
                                                            }`}
                                                        >
                                                            {key.toUpperCase()}
                                                        </span>
                                                        <p className="mt-1 text-gray-600 dark:text-gray-400">
                                                            {value}
                                                        </p>
                                                    </div>
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                </div>
                            )}

                            {/* Navigation buttons */}
                            <div className="flex justify-between mt-8">
                                <button
                                    type="button"
                                    onClick={prevQuestion}
                                    disabled={currentQuestion === 0}
                                    className="px-4 py-2 bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 rounded-md disabled:opacity-50"
                                >
                                    Previous
                                </button>

                                {currentQuestion < questions.length - 1 ? (
                                    <button
                                        type="button"
                                        onClick={nextQuestion}
                                        className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 transition"
                                    >
                                        Next
                                    </button>
                                ) : (
                                    <button
                                        type="button"
                                        onClick={submitTest}
                                        disabled={
                                            isSubmitting || isAutoSubmitting
                                        }
                                        className="flex items-center px-4 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 transition disabled:opacity-75 disabled:bg-green-500"
                                    >
                                        {isSubmitting || isAutoSubmitting ? (
                                            <>
                                                <svg
                                                    className="animate-spin -ml-1 mr-2 h-4 w-4 text-white"
                                                    xmlns="http://www.w3.org/2000/svg"
                                                    fill="none"
                                                    viewBox="0 0 24 24"
                                                >
                                                    <circle
                                                        className="opacity-25"
                                                        cx="12"
                                                        cy="12"
                                                        r="10"
                                                        stroke="currentColor"
                                                        strokeWidth="4"
                                                    ></circle>
                                                    <path
                                                        className="opacity-75"
                                                        fill="currentColor"
                                                        d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                                                    ></path>
                                                </svg>
                                                {isAutoSubmitting
                                                    ? "Auto-Submitting..."
                                                    : "Submitting..."}
                                            </>
                                        ) : (
                                            <>
                                                Submit Test
                                                <Send className="ml-2 h-4 w-4" />
                                            </>
                                        )}
                                    </button>
                                )}
                            </div>

                            {/* Auto-submission warning for low time */}
                            {warningLevel === "high" && (
                                <div className="mt-6 p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-md">
                                    <h3 className="text-md font-semibold text-red-700 dark:text-red-300 mb-1 flex items-center">
                                        <AlertTriangle className="h-4 w-4 mr-1" />
                                        Auto-Submission Imminent
                                    </h3>
                                    <p className="text-sm text-red-600 dark:text-red-400">
                                        Your test will be automatically
                                        submitted when the timer reaches zero.
                                        Please finalize your answers
                                        immediately.
                                    </p>
                                </div>
                            )}

                            {/* Help text */}
                            <div className="mt-8 pt-4 border-t border-gray-200 dark:border-gray-700">
                                <div className="flex items-start">
                                    <div className="flex-shrink-0">
                                        <HelpCircle className="h-5 w-5 text-gray-400 dark:text-gray-500" />
                                    </div>
                                    <div className="ml-3">
                                        <h3 className="text-sm font-medium text-gray-600 dark:text-gray-400">
                                            Test Instructions
                                        </h3>
                                        <ul className="mt-1 text-sm text-gray-500 dark:text-gray-500 list-disc list-inside space-y-1">
                                            <li>
                                                Answer all questions to the best
                                                of your ability.
                                            </li>
                                            <li>
                                                You can navigate between
                                                questions using the numbered
                                                buttons or next/previous
                                                buttons.
                                            </li>
                                            <li>
                                                Your answers are saved as you
                                                select them.
                                            </li>
                                            <li>
                                                The test will be automatically
                                                submitted when time expires.
                                            </li>
                                        </ul>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </WriterLayout>
    );
}
