import { clsx } from "clsx";
import { twMerge } from "tailwind-merge";

/**
 * Combines multiple class names with Tailwind CSS optimization
 * @param {...string} inputs - Class names to be combined
 * @returns {string} - Combined class names
 */
export function cn(...inputs) {
    return twMerge(clsx(inputs));
}

/**
 * Format date to a human-readable string
 * @param {string|Date} date - Date to format
 * @param {object} options - Formatting options
 * @returns {string} - Formatted date string
 */
export function formatDate(date, options = {}) {
    if (!date) return "";

    const defaultOptions = {
        dateStyle: "medium",
        timeStyle: undefined,
    };

    const mergedOptions = { ...defaultOptions, ...options };

    try {
        const dateObj = typeof date === "string" ? new Date(date) : date;
        return new Intl.DateTimeFormat("en-US", mergedOptions).format(dateObj);
    } catch (error) {
        console.error("Error formatting date:", error);
        return date?.toString() || "";
    }
}

/**
 * Format currency value
 * @param {number} amount - Amount to format
 * @param {string} currency - Currency code (default: USD)
 * @returns {string} - Formatted currency string
 */
export function formatCurrency(amount, currency = "USD") {
    return new Intl.NumberFormat("en-US", {
        style: "currency",
        currency,
    }).format(amount);
}

/**
 * Format date for HTML input elements (date, datetime-local)
 * @param {string|Date} date - Date to format
 * @param {boolean} includeTime - Whether to include time component
 * @returns {string} - Date formatted for input field
 */
export function formatDateForInput(date, includeTime = true) {
    if (!date) return "";

    try {
        const dateObj = typeof date === "string" ? new Date(date) : date;

        if (isNaN(dateObj.getTime())) {
            return "";
        }

        // Format date part: YYYY-MM-DD
        const year = dateObj.getFullYear();
        const month = String(dateObj.getMonth() + 1).padStart(2, "0");
        const day = String(dateObj.getDate()).padStart(2, "0");

        const datePart = `${year}-${month}-${day}`;

        if (!includeTime) {
            return datePart;
        }

        // Format time part: HH:MM
        const hours = String(dateObj.getHours()).padStart(2, "0");
        const minutes = String(dateObj.getMinutes()).padStart(2, "0");

        // Combine for datetime-local input format: YYYY-MM-DDThh:mm
        return `${datePart}T${hours}:${minutes}`;
    } catch (error) {
        console.error("Error formatting date for input:", error);
        return "";
    }
}
