/**
 * Comprehensive Blog Post Validation System
 * Validates blog post data structure, content, and SEO requirements
 */

class BlogPostValidator {
    constructor(options = {}) {
        this.options = {
            strictMode: false,
            checkDuplicateSlugs: true,
            validateImages: true,
            validateSEO: true,
            maxTitleLength: 100,
            minTitleLength: 10,
            maxExcerptLength: 300,
            minExcerptLength: 50,
            maxContentLength: 50000,
            minContentLength: 500,
            maxTagsCount: 10,
            maxKeywordsCount: 20,
            requiredCategories: [
                "essay-writing",
                "research-papers",
                "academic-writing",
                "study-tips",
                "case-studies",
                "literature-review",
                "assignments",
                "dissertation-writing",
                "phd-writing",
            ],
            ...options,
        };

        this.errors = [];
        this.warnings = [];
        this.suggestions = [];
    }

    /**
     * Main validation method
     * @param {Object} postData - Blog post data to validate
     * @param {Object} context - Additional context (existing posts, etc.)
     * @returns {Object} - Validation result
     */
    validate(postData, context = {}) {
        this.reset();

        // Structure validation
        this.validateStructure(postData);

        // Required fields validation
        this.validateRequiredFields(postData);

        // Data type validation
        this.validateDataTypes(postData);

        // Content validation
        this.validateContent(postData);

        // SEO validation
        if (this.options.validateSEO) {
            this.validateSEO(postData);
        }

        // Slug validation
        this.validateSlug(postData, context);

        // Category validation
        this.validateCategory(postData);

        // Tags and keywords validation
        this.validateTagsAndKeywords(postData);

        // Date validation
        this.validateDates(postData);

        // Image validation
        if (this.options.validateImages) {
            this.validateImages(postData);
        }

        // Content quality checks
        this.validateContentQuality(postData);

        // Generate suggestions
        this.generateSuggestions(postData);

        return {
            isValid: this.errors.length === 0,
            hasWarnings: this.warnings.length > 0,
            errors: this.errors,
            warnings: this.warnings,
            suggestions: this.suggestions,
            score: this.calculateScore(postData),
        };
    }

    /**
     * Reset validation state
     */
    reset() {
        this.errors = [];
        this.warnings = [];
        this.suggestions = [];
    }

    /**
     * Validate basic structure
     */
    validateStructure(postData) {
        if (!postData || typeof postData !== "object") {
            this.addError("INVALID_STRUCTURE", "Post data must be an object");
            return;
        }

        if (Array.isArray(postData)) {
            this.addError("INVALID_STRUCTURE", "Post data cannot be an array");
            return;
        }

        // Check for circular references
        try {
            JSON.stringify(postData);
        } catch (error) {
            this.addError(
                "CIRCULAR_REFERENCE",
                "Post data contains circular references"
            );
        }
    }

    /**
     * Validate required fields
     */
    validateRequiredFields(postData) {
        const requiredFields = [
            "title",
            "slug",
            "excerpt",
            "content",
            "author",
            "category",
            "publishedAt",
        ];

        requiredFields.forEach((field) => {
            if (!postData[field]) {
                this.addError(
                    "MISSING_REQUIRED_FIELD",
                    `Required field '${field}' is missing`
                );
            } else if (
                typeof postData[field] === "string" &&
                postData[field].trim() === ""
            ) {
                this.addError(
                    "EMPTY_REQUIRED_FIELD",
                    `Required field '${field}' is empty`
                );
            }
        });
    }

    /**
     * Validate data types
     */
    validateDataTypes(postData) {
        const typeValidations = {
            title: "string",
            slug: "string",
            excerpt: "string",
            content: "string",
            author: "string",
            category: "string",
            publishedAt: "string",
            readTime: "number",
            featured: "boolean",
            tags: "array",
            keywords: "array",
            meta: "object",
            seo: "object",
        };

        Object.entries(typeValidations).forEach(([field, expectedType]) => {
            if (postData[field] !== undefined) {
                const actualType = this.getDataType(postData[field]);
                if (actualType !== expectedType) {
                    this.addError(
                        "INVALID_DATA_TYPE",
                        `Field '${field}' must be of type ${expectedType}, got ${actualType}`
                    );
                }
            }
        });
    }

    /**
     * Validate content fields
     */
    validateContent(postData) {
        // Title validation
        if (postData.title) {
            if (postData.title.length < this.options.minTitleLength) {
                this.addWarning(
                    "TITLE_TOO_SHORT",
                    `Title is too short (${postData.title.length} characters). Minimum: ${this.options.minTitleLength}`
                );
            }
            if (postData.title.length > this.options.maxTitleLength) {
                this.addWarning(
                    "TITLE_TOO_LONG",
                    `Title is too long (${postData.title.length} characters). Maximum: ${this.options.maxTitleLength}`
                );
            }
        }

        // Excerpt validation
        if (postData.excerpt) {
            if (postData.excerpt.length < this.options.minExcerptLength) {
                this.addWarning(
                    "EXCERPT_TOO_SHORT",
                    `Excerpt is too short (${postData.excerpt.length} characters). Minimum: ${this.options.minExcerptLength}`
                );
            }
            if (postData.excerpt.length > this.options.maxExcerptLength) {
                this.addWarning(
                    "EXCERPT_TOO_LONG",
                    `Excerpt is too long (${postData.excerpt.length} characters). Maximum: ${this.options.maxExcerptLength}`
                );
            }
        }

        // Content validation
        if (postData.content) {
            if (postData.content.length < this.options.minContentLength) {
                this.addWarning(
                    "CONTENT_TOO_SHORT",
                    `Content is too short (${postData.content.length} characters). Minimum: ${this.options.minContentLength}`
                );
            }
            if (postData.content.length > this.options.maxContentLength) {
                this.addWarning(
                    "CONTENT_TOO_LONG",
                    `Content is too long (${postData.content.length} characters). Maximum: ${this.options.maxContentLength}`
                );
            }
        }

        // Read time validation
        if (postData.readTime !== undefined) {
            if (postData.readTime < 1) {
                this.addWarning(
                    "INVALID_READ_TIME",
                    "Read time must be at least 1 minute"
                );
            }
            if (postData.readTime > 60) {
                this.addWarning(
                    "EXCESSIVE_READ_TIME",
                    "Read time seems excessive (over 60 minutes)"
                );
            }
        }
    }

    /**
     * Validate SEO fields
     */
    validateSEO(postData) {
        if (postData.meta) {
            // SEO title
            if (!postData.meta.title) {
                this.addWarning(
                    "MISSING_SEO_TITLE",
                    "SEO title is recommended for better search visibility"
                );
            } else if (postData.meta.title.length > 60) {
                this.addWarning(
                    "SEO_TITLE_TOO_LONG",
                    `SEO title is too long (${postData.meta.title.length} characters). Maximum recommended: 60`
                );
            }

            // SEO description
            if (!postData.meta.description) {
                this.addWarning(
                    "MISSING_SEO_DESCRIPTION",
                    "SEO description is recommended for better search visibility"
                );
            } else if (postData.meta.description.length > 160) {
                this.addWarning(
                    "SEO_DESCRIPTION_TOO_LONG",
                    `SEO description is too long (${postData.meta.description.length} characters). Maximum recommended: 160`
                );
            } else if (postData.meta.description.length < 50) {
                this.addWarning(
                    "SEO_DESCRIPTION_TOO_SHORT",
                    `SEO description is too short (${postData.meta.description.length} characters). Minimum recommended: 50`
                );
            }

            // SEO keywords
            if (!postData.meta.keywords) {
                this.addWarning(
                    "MISSING_SEO_KEYWORDS",
                    "SEO keywords are recommended for better search visibility"
                );
            }

            // Canonical URL
            if (!postData.meta.canonical) {
                this.addWarning(
                    "MISSING_CANONICAL_URL",
                    "Canonical URL is recommended for SEO"
                );
            }
        } else {
            this.addWarning(
                "MISSING_SEO_META",
                "SEO metadata is recommended for better search visibility"
            );
        }

        // Open Graph validation
        if (postData.seo && postData.seo.openGraph) {
            if (!postData.seo.openGraph.title) {
                this.addWarning(
                    "MISSING_OG_TITLE",
                    "Open Graph title is recommended for social sharing"
                );
            }
            if (!postData.seo.openGraph.description) {
                this.addWarning(
                    "MISSING_OG_DESCRIPTION",
                    "Open Graph description is recommended for social sharing"
                );
            }
            if (!postData.seo.openGraph.image) {
                this.addWarning(
                    "MISSING_OG_IMAGE",
                    "Open Graph image is recommended for social sharing"
                );
            }
        }
    }

    /**
     * Validate slug
     */
    validateSlug(postData, context) {
        if (!postData.slug) return;

        // Format validation
        const slugRegex = /^[a-z0-9]+(?:-[a-z0-9]+)*$/;
        if (!slugRegex.test(postData.slug)) {
            this.addError(
                "INVALID_SLUG_FORMAT",
                "Slug must contain only lowercase letters, numbers, and hyphens"
            );
        }

        // Length validation
        if (postData.slug.length < 3) {
            this.addWarning(
                "SLUG_TOO_SHORT",
                "Slug should be at least 3 characters long"
            );
        }
        if (postData.slug.length > 100) {
            this.addWarning(
                "SLUG_TOO_LONG",
                "Slug should be less than 100 characters long"
            );
        }

        // Duplicate check
        if (this.options.checkDuplicateSlugs && context.existingSlugs) {
            if (context.existingSlugs.includes(postData.slug)) {
                this.addError("DUPLICATE_SLUG", "This slug already exists");
            }
        }

        // Reserved words check
        const reservedWords = [
            "admin",
            "api",
            "blog",
            "dashboard",
            "login",
            "register",
            "search",
        ];
        if (reservedWords.includes(postData.slug)) {
            this.addError(
                "RESERVED_SLUG",
                "This slug is reserved and cannot be used"
            );
        }
    }

    /**
     * Validate category
     */
    validateCategory(postData) {
        if (!postData.category) return;

        if (!this.options.requiredCategories.includes(postData.category)) {
            this.addError(
                "INVALID_CATEGORY",
                `Category '${
                    postData.category
                }' is not valid. Must be one of: ${this.options.requiredCategories.join(
                    ", "
                )}`
            );
        }
    }

    /**
     * Validate tags and keywords
     */
    validateTagsAndKeywords(postData) {
        // Tags validation
        if (postData.tags) {
            if (!Array.isArray(postData.tags)) {
                this.addError("INVALID_TAGS_FORMAT", "Tags must be an array");
            } else {
                if (postData.tags.length > this.options.maxTagsCount) {
                    this.addWarning(
                        "TOO_MANY_TAGS",
                        `Too many tags (${postData.tags.length}). Maximum recommended: ${this.options.maxTagsCount}`
                    );
                }

                postData.tags.forEach((tag, index) => {
                    if (typeof tag !== "string") {
                        this.addError(
                            "INVALID_TAG_TYPE",
                            `Tag at index ${index} must be a string`
                        );
                    } else if (tag.trim() === "") {
                        this.addError(
                            "EMPTY_TAG",
                            `Tag at index ${index} is empty`
                        );
                    }
                });
            }
        }

        // Keywords validation
        if (postData.keywords) {
            if (!Array.isArray(postData.keywords)) {
                this.addError(
                    "INVALID_KEYWORDS_FORMAT",
                    "Keywords must be an array"
                );
            } else {
                if (postData.keywords.length > this.options.maxKeywordsCount) {
                    this.addWarning(
                        "TOO_MANY_KEYWORDS",
                        `Too many keywords (${postData.keywords.length}). Maximum recommended: ${this.options.maxKeywordsCount}`
                    );
                }

                postData.keywords.forEach((keyword, index) => {
                    if (typeof keyword !== "string") {
                        this.addError(
                            "INVALID_KEYWORD_TYPE",
                            `Keyword at index ${index} must be a string`
                        );
                    } else if (keyword.trim() === "") {
                        this.addError(
                            "EMPTY_KEYWORD",
                            `Keyword at index ${index} is empty`
                        );
                    }
                });
            }
        }
    }

    /**
     * Validate dates
     */
    validateDates(postData) {
        if (postData.publishedAt) {
            const publishedDate = new Date(postData.publishedAt);
            if (isNaN(publishedDate.getTime())) {
                this.addError(
                    "INVALID_PUBLISHED_DATE",
                    "Published date is not a valid date"
                );
            } else {
                const now = new Date();
                if (publishedDate > now) {
                    this.addWarning(
                        "FUTURE_PUBLISHED_DATE",
                        "Published date is in the future"
                    );
                }
            }
        }

        if (postData.updatedAt) {
            const updatedDate = new Date(postData.updatedAt);
            if (isNaN(updatedDate.getTime())) {
                this.addError(
                    "INVALID_UPDATED_DATE",
                    "Updated date is not a valid date"
                );
            } else {
                const now = new Date();
                if (updatedDate > now) {
                    this.addWarning(
                        "FUTURE_UPDATED_DATE",
                        "Updated date is in the future"
                    );
                }

                // Check if updated date is before published date
                if (postData.publishedAt) {
                    const publishedDate = new Date(postData.publishedAt);
                    if (
                        !isNaN(publishedDate.getTime()) &&
                        updatedDate < publishedDate
                    ) {
                        this.addError(
                            "UPDATED_BEFORE_PUBLISHED",
                            "Updated date cannot be before published date"
                        );
                    }
                }
            }
        }
    }

    /**
     * Validate images
     */
    validateImages(postData) {
        // Check for image references in content
        if (postData.content) {
            const imageRegex = /!\[([^\]]*)\]\(([^)]+)\)/g;
            let match;
            const images = [];

            while ((match = imageRegex.exec(postData.content)) !== null) {
                images.push({
                    alt: match[1],
                    src: match[2],
                });
            }

            images.forEach((image, index) => {
                if (!image.alt || image.alt.trim() === "") {
                    this.addWarning(
                        "MISSING_IMAGE_ALT",
                        `Image ${index + 1} is missing alt text`
                    );
                }

                if (!image.src || image.src.trim() === "") {
                    this.addError(
                        "MISSING_IMAGE_SRC",
                        `Image ${index + 1} is missing source URL`
                    );
                }

                // Check for relative vs absolute URLs
                if (
                    image.src &&
                    !image.src.startsWith("http") &&
                    !image.src.startsWith("/")
                ) {
                    this.addWarning(
                        "RELATIVE_IMAGE_PATH",
                        `Image ${index + 1} uses relative path: ${image.src}`
                    );
                }
            });
        }

        // Check featured image
        if (postData.featuredImage) {
            if (typeof postData.featuredImage === "string") {
                if (!postData.featuredImage.trim()) {
                    this.addError(
                        "EMPTY_FEATURED_IMAGE",
                        "Featured image URL is empty"
                    );
                }
            } else if (typeof postData.featuredImage === "object") {
                if (!postData.featuredImage.src) {
                    this.addError(
                        "MISSING_FEATURED_IMAGE_SRC",
                        "Featured image is missing source URL"
                    );
                }
                if (!postData.featuredImage.alt) {
                    this.addWarning(
                        "MISSING_FEATURED_IMAGE_ALT",
                        "Featured image is missing alt text"
                    );
                }
            }
        }
    }

    /**
     * Validate content quality
     */
    validateContentQuality(postData) {
        if (!postData.content) return;

        // Check for common quality issues
        const content = postData.content;

        // Check for excessive repetition
        const words = content.toLowerCase().split(/\s+/);
        const wordCount = {};
        words.forEach((word) => {
            if (word.length > 3) {
                wordCount[word] = (wordCount[word] || 0) + 1;
            }
        });

        Object.entries(wordCount).forEach(([word, count]) => {
            if (count > words.length * 0.05) {
                this.addWarning(
                    "EXCESSIVE_WORD_REPETITION",
                    `Word "${word}" appears ${count} times (${(
                        (count / words.length) *
                        100
                    ).toFixed(1)}% of content)`
                );
            }
        });

        // Check for too many short paragraphs
        const paragraphs = content.split("\n\n").filter((p) => p.trim());
        const shortParagraphs = paragraphs.filter(
            (p) => p.split(" ").length < 10
        );
        if (shortParagraphs.length > paragraphs.length * 0.7) {
            this.addWarning(
                "TOO_MANY_SHORT_PARAGRAPHS",
                "Content has many short paragraphs. Consider combining some for better readability"
            );
        }

        // Check for missing headings
        const headingRegex = /^#{1,6}\s/gm;
        const headings = content.match(headingRegex);
        if (!headings || headings.length === 0) {
            this.addWarning(
                "NO_HEADINGS",
                "Content has no headings. Consider adding structure with H2, H3, etc."
            );
        }

        // Check for very long paragraphs
        paragraphs.forEach((paragraph, index) => {
            if (paragraph.split(" ").length > 100) {
                this.addWarning(
                    "LONG_PARAGRAPH",
                    `Paragraph ${index + 1} is very long (${
                        paragraph.split(" ").length
                    } words). Consider breaking it up`
                );
            }
        });

        // Check for code blocks without syntax highlighting
        const codeBlockRegex = /```([^`]*?)```/g;
        let codeMatch;
        while ((codeMatch = codeBlockRegex.exec(content)) !== null) {
            const codeBlock = codeMatch[1];
            if (
                !codeBlock.trim().startsWith("javascript") &&
                !codeBlock.trim().startsWith("js") &&
                !codeBlock.trim().startsWith("php") &&
                !codeBlock.trim().startsWith("python") &&
                !codeBlock.trim().startsWith("css") &&
                !codeBlock.trim().startsWith("html")
            ) {
                this.addSuggestion(
                    "CODE_SYNTAX_HIGHLIGHTING",
                    "Consider adding syntax highlighting to code blocks"
                );
            }
        }
    }

    /**
     * Generate suggestions for improvement
     */
    generateSuggestions(postData) {
        // Reading time suggestions
        if (postData.content && !postData.readTime) {
            const words = postData.content.split(/\s+/).length;
            const estimatedReadTime = Math.ceil(words / 200);
            this.addSuggestion(
                "CALCULATE_READ_TIME",
                `Consider adding readTime: ${estimatedReadTime} (estimated based on ${words} words)`
            );
        }

        // Featured post suggestion
        if (postData.featured === undefined) {
            this.addSuggestion(
                "SET_FEATURED_STATUS",
                "Consider setting featured status (true/false)"
            );
        }

        // Tags suggestion
        if (!postData.tags || postData.tags.length === 0) {
            this.addSuggestion(
                "ADD_TAGS",
                "Consider adding tags to improve discoverability"
            );
        }

        // Keywords suggestion
        if (!postData.keywords || postData.keywords.length === 0) {
            this.addSuggestion(
                "ADD_KEYWORDS",
                "Consider adding keywords for better SEO"
            );
        }

        // Social media optimization
        if (!postData.seo || !postData.seo.openGraph) {
            this.addSuggestion(
                "ADD_OPEN_GRAPH",
                "Consider adding Open Graph metadata for social media sharing"
            );
        }

        // Internal linking suggestions
        if (postData.content && !postData.content.includes("](")) {
            this.addSuggestion(
                "ADD_INTERNAL_LINKS",
                "Consider adding internal links to other relevant posts"
            );
        }

        // Call-to-action suggestions
        if (
            postData.content &&
            !postData.content.toLowerCase().includes("subscribe") &&
            !postData.content.toLowerCase().includes("share") &&
            !postData.content.toLowerCase().includes("comment")
        ) {
            this.addSuggestion(
                "ADD_CTA",
                "Consider adding a call-to-action (subscribe, share, comment)"
            );
        }
    }

    /**
     * Calculate quality score
     */
    calculateScore(postData) {
        let score = 100;

        // Deduct points for errors
        score -= this.errors.length * 10;

        // Deduct points for warnings
        score -= this.warnings.length * 5;

        // Bonus points for good practices
        if (postData.meta && postData.meta.title) score += 5;
        if (postData.meta && postData.meta.description) score += 5;
        if (postData.meta && postData.meta.keywords) score += 5;
        if (postData.tags && postData.tags.length > 0) score += 5;
        if (postData.keywords && postData.keywords.length > 0) score += 5;
        if (postData.featuredImage) score += 5;
        if (postData.readTime) score += 5;
        if (postData.seo && postData.seo.openGraph) score += 10;

        // Content quality bonuses
        if (postData.content) {
            const headingRegex = /^#{1,6}\s/gm;
            const headings = postData.content.match(headingRegex);
            if (headings && headings.length > 0) score += 5;

            const linkRegex = /\[([^\]]+)\]\(([^)]+)\)/g;
            const links = postData.content.match(linkRegex);
            if (links && links.length > 0) score += 5;

            const imageRegex = /!\[([^\]]*)\]\(([^)]+)\)/g;
            const images = postData.content.match(imageRegex);
            if (images && images.length > 0) score += 5;
        }

        return Math.max(0, Math.min(100, score));
    }

    /**
     * Add error
     */
    addError(code, message) {
        this.errors.push({
            code,
            message,
            severity: "error",
        });
    }

    /**
     * Add warning
     */
    addWarning(code, message) {
        this.warnings.push({
            code,
            message,
            severity: "warning",
        });
    }

    /**
     * Add suggestion
     */
    addSuggestion(code, message) {
        this.suggestions.push({
            code,
            message,
            severity: "info",
        });
    }

    /**
     * Get data type
     */
    getDataType(value) {
        if (Array.isArray(value)) return "array";
        if (value === null) return "null";
        return typeof value;
    }

    /**
     * Validate batch of posts
     */
    validateBatch(posts, context = {}) {
        const results = [];
        const allSlugs = new Set();

        posts.forEach((post, index) => {
            const postContext = {
                ...context,
                existingSlugs: [
                    ...(context.existingSlugs || []),
                    ...Array.from(allSlugs),
                ],
            };

            const result = this.validate(post, postContext);
            results.push({
                index,
                post,
                result,
            });

            // Add slug to set for duplicate checking
            if (post.slug) {
                allSlugs.add(post.slug);
            }
        });

        return {
            results,
            totalPosts: posts.length,
            validPosts: results.filter((r) => r.result.isValid).length,
            invalidPosts: results.filter((r) => !r.result.isValid).length,
            postsWithWarnings: results.filter((r) => r.result.hasWarnings)
                .length,
            averageScore:
                results.reduce((sum, r) => sum + r.result.score, 0) /
                results.length,
        };
    }

    /**
     * Get validation summary
     */
    getValidationSummary(results) {
        const summary = {
            totalErrors: 0,
            totalWarnings: 0,
            totalSuggestions: 0,
            errorsByType: {},
            warningsByType: {},
            suggestionsByType: {},
        };

        results.forEach((result) => {
            summary.totalErrors += result.result.errors.length;
            summary.totalWarnings += result.result.warnings.length;
            summary.totalSuggestions += result.result.suggestions.length;

            result.result.errors.forEach((error) => {
                summary.errorsByType[error.code] =
                    (summary.errorsByType[error.code] || 0) + 1;
            });

            result.result.warnings.forEach((warning) => {
                summary.warningsByType[warning.code] =
                    (summary.warningsByType[warning.code] || 0) + 1;
            });

            result.result.suggestions.forEach((suggestion) => {
                summary.suggestionsByType[suggestion.code] =
                    (summary.suggestionsByType[suggestion.code] || 0) + 1;
            });
        });

        return summary;
    }
}

// Export validation utility functions
export const validateBlogPost = (postData, options = {}) => {
    const validator = new BlogPostValidator(options);
    return validator.validate(postData);
};

export const validateBlogPosts = (posts, options = {}) => {
    const validator = new BlogPostValidator(options);
    return validator.validateBatch(posts);
};

export const createValidator = (options = {}) => {
    return new BlogPostValidator(options);
};

export default BlogPostValidator;
