/**
 * Image optimization utilities
 */

/**
 * Get optimized image URLs for a given source
 * @param {string} src - Original image source
 * @returns {object} Object with webp and fallback URLs
 */
export const getOptimizedImageUrls = (src) => {
    if (!src) return { webp: null, fallback: src };

    const pathInfo = src.split(".");
    const extension = pathInfo.pop();
    const basePath = pathInfo.join(".");

    return {
        webp: `${basePath}.webp`,
        fallback: src,
    };
};

/**
 * Generate responsive image sizes attribute
 * @param {object} breakpoints - Breakpoint configuration
 * @returns {string} Sizes attribute value
 */
export const generateSizes = (breakpoints = {}) => {
    const defaultBreakpoints = {
        sm: "640px",
        md: "768px",
        lg: "1024px",
        xl: "1280px",
        "2xl": "1536px",
    };

    const merged = { ...defaultBreakpoints, ...breakpoints };

    // Generate sizes string
    const sizes = Object.entries(merged)
        .map(([key, value]) => `(max-width: ${value}) 100vw`)
        .join(", ");

    return `${sizes}, 100vw`;
};

/**
 * Check if WebP is supported
 * @returns {Promise<boolean>}
 */
export const supportsWebP = () => {
    return new Promise((resolve) => {
        if (typeof window === "undefined") {
            resolve(false);
            return;
        }

        const webP = new Image();
        webP.onload = webP.onerror = () => {
            resolve(webP.height === 2);
        };
        webP.src =
            "data:image/webp;base64,UklGRjoAAABXRUJQVlA4IC4AAACyAgCdASoCAAIALmk0mk0iIiIiIgBoSygABc6WWgAA/veff/0PP8bA//LwYAAA";
    });
};

/**
 * Check if AVIF is supported
 * @returns {Promise<boolean>}
 */
export const supportsAVIF = () => {
    return new Promise((resolve) => {
        if (typeof window === "undefined") {
            resolve(false);
            return;
        }

        const avif = new Image();
        avif.onload = avif.onerror = () => {
            resolve(avif.height === 2);
        };
        avif.src =
            "data:image/avif;base64,AAAAIGZ0eXBhdmlmAAAAAGF2aWZtaWYxbWlhZk1BMUIAAADybWV0YQAAAAAAAAAoaGRscgAAAAAAAAAAcGljdAAAAAAAAAAAAAAAAGxpYmF2aWYAAAAADnBpdG0AAAAAAAEAAAAeaWxvYwAAAABEAAABAAEAAAABAAABGgAAAB0AAAAoaWluZgAAAAAAAQAAABppbmZlAgAAAAABAABhdjAxQ29sb3IAAAAAamlwcnAAAABLaXBjbwAAABRpc3BlAAAAAAAAAAIAAAACAAAAEHBpeGkAAAAAAwgICAAAAAxhdjFDgQ0MAAAAABNjb2xybmNseAACAAIAAYAAAAAXaXBtYQAAAAAAAAABAAEEAQKDBAAAACVtZGF0EgAKCBgABogQEAwgMg8f8D///8WfhwB8+ErK42A=";
    });
};

/**
 * Get the best supported image format
 * @returns {Promise<string>} Returns 'avif', 'webp', or 'fallback'
 */
export const getBestImageFormat = async () => {
    if (await supportsAVIF()) return "avif";
    if (await supportsWebP()) return "webp";
    return "fallback";
};

/**
 * Create a responsive image srcset
 * @param {string} basePath - Base image path without extension
 * @param {array} sizes - Array of sizes [width, height]
 * @param {string} format - Image format (webp, jpg, png)
 * @returns {string} srcset string
 */
export const createSrcSet = (basePath, sizes, format = "webp") => {
    return sizes
        .map(
            ([width, height]) =>
                `${basePath}-${width}x${height}.${format} ${width}w`
        )
        .join(", ");
};

/**
 * Lazy load images with intersection observer
 * @param {string} selector - CSS selector for images to lazy load
 */
export const lazyLoadImages = (selector = "img[data-src]") => {
    if (typeof window === "undefined" || !("IntersectionObserver" in window)) {
        return;
    }

    const images = document.querySelectorAll(selector);

    const imageObserver = new IntersectionObserver((entries, observer) => {
        entries.forEach((entry) => {
            if (entry.isIntersecting) {
                const img = entry.target;
                img.src = img.dataset.src;
                img.classList.remove("lazy");
                imageObserver.unobserve(img);
            }
        });
    });

    images.forEach((img) => imageObserver.observe(img));
};
