/**
 * Order State Manager - Handles localStorage persistence with validation and error recovery
 */

const STORAGE_KEY = 'orderFormData';
const MAX_AGE_HOURS = 24; // 24 hours
const VERSION = '1.0'; // For future migrations

export class OrderStateManager {
    /**
     * Save order state to localStorage with validation
     */
    static saveOrderState(state) {
        try {
            const dataToSave = {
                ...state,
                version: VERSION,
                timestamp: Date.now(),
                lastSaved: new Date().toISOString()
            };

            // Validate required fields
            if (!this.validateState(state)) {
                console.warn('Invalid order state, skipping save');
                return false;
            }

            localStorage.setItem(STORAGE_KEY, JSON.stringify(dataToSave));
            return true;
        } catch (error) {
            console.error('Failed to save order state:', error);
            return false;
        }
    }

    /**
     * Load order state from localStorage with validation
     */
    static loadOrderState() {
        try {
            const stored = localStorage.getItem(STORAGE_KEY);
            if (!stored) return null;

            const parsed = JSON.parse(stored);
            
            // Check if data is too old
            if (this.isDataExpired(parsed.timestamp)) {
                this.clearOrderState();
                return null;
            }

            // Validate loaded data
            if (!this.validateLoadedState(parsed)) {
                // Silently clear invalid state - will fall back to fresh form
                this.clearOrderState();
                return null;
            }

            return parsed;
        } catch (error) {
            console.error('Failed to load order state:', error);
            this.clearOrderState();
            return null;
        }
    }

    /**
     * Clear order state from localStorage
     */
    static clearOrderState() {
        try {
            localStorage.removeItem(STORAGE_KEY);
            return true;
        } catch (error) {
            console.error('Failed to clear order state:', error);
            return false;
        }
    }

    /**
     * Check if stored data is expired
     */
    static isDataExpired(timestamp) {
        const maxAge = MAX_AGE_HOURS * 60 * 60 * 1000; // Convert to milliseconds
        return Date.now() - timestamp > maxAge;
    }

    /**
     * Validate current state before saving
     */
    static validateState(state) {
        // Basic validation - ensure we have at least some form data
        if (!state || typeof state !== 'object') {
            return false;
        }

        // More lenient validation - allow partial data
        const hasAnyData = state.title || state.subject || state.aclevel || state.papertype || state.pages || state.deadline || state.spacing || state.currency || state.instructions;
        if (!hasAnyData) {
            return false;
        }

        // Validate step (allow 1-5 range)
        if (state.currentStep && (state.currentStep < 1 || state.currentStep > 5)) {
            return false;
        }

        return true;
    }

    /**
     * Validate loaded state from localStorage
     */
    static validateLoadedState(parsed) {
        if (!parsed || typeof parsed !== 'object') {
            return false;
        }

        // Check version compatibility (for future migrations)
        if (parsed.version && parsed.version !== VERSION) {
            console.warn('Order state version mismatch, clearing old data');
            return false;
        }

        // Check timestamp
        if (!parsed.timestamp || this.isDataExpired(parsed.timestamp)) {
            return false;
        }

        // Basic structure validation - accept both wrapped (parsed.data) and unwrapped formats
        // Also check if we have any order-related fields directly in parsed
        const hasWrappedData = parsed.data && typeof parsed.data === 'object';
        const hasDirectFields = parsed.title || parsed.subject || parsed.aclevel || 
                               parsed.papertype || parsed.pages || parsed.deadline || 
                               parsed.spacing || parsed.instructions || parsed.currentStep;
        
        if (!hasWrappedData && !hasDirectFields) {
            return false;
        }

        return true;
    }

    /**
     * Get state summary for debugging
     */
    static getStateSummary() {
        try {
            const stored = localStorage.getItem(STORAGE_KEY);
            if (!stored) return null;

            const parsed = JSON.parse(stored);
            return {
                hasData: !!parsed,
                timestamp: parsed?.timestamp,
                age: parsed?.timestamp ? Date.now() - parsed.timestamp : null,
                step: parsed?.currentStep,
                version: parsed?.version,
                isExpired: parsed?.timestamp ? this.isDataExpired(parsed.timestamp) : true
            };
        } catch (error) {
            return { error: error.message };
        }
    }

    /**
     * Migrate old state format to new format
     */
    static migrateState(oldState) {
        // Handle migration from old format to new format
        if (oldState && !oldState.version) {
            // Old format - migrate to new format
            return {
                ...oldState,
                version: VERSION,
                timestamp: oldState.timestamp || Date.now(),
                lastSaved: new Date().toISOString()
            };
        }
        return oldState;
    }
} 