<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Order;
use App\Models\OrderFile;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Notification;

class WriterFileUploadTest extends TestCase
{
    use RefreshDatabase;

    protected $writer;
    protected $client;
    protected $order;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create test users
        $this->writer = User::factory()->create([
            'user_type' => 'writer',
            'name' => 'Test Writer'
        ]);

        $this->client = User::factory()->create([
            'user_type' => 'client',
            'name' => 'Test Client'
        ]);

        // Create a test order
        $this->order = Order::factory()->create([
            'user_id' => $this->client->id,
            'writer_id' => $this->writer->id,
            'order_status' => 'in_progress',
            'order_number' => 'TEST-001'
        ]);

        // Mock storage
        Storage::fake('public');
        
        // Mock notifications
        Notification::fake();
    }

    /** @test */
    public function writer_can_upload_draft_file()
    {
        $file = UploadedFile::fake()->create('draft.docx', 1024, 'application/vnd.openxmlformats-officedocument.wordprocessingml.document');

        $response = $this->actingAs($this->writer)
            ->postJson('/order-files/writer-upload', [
                'file' => $file,
                'order_id' => $this->order->id,
                'file_category' => 'draft',
                'description' => 'Work in progress',
                'uploader_user_type' => 'writer'
            ]);

        $response->assertStatus(200)
            ->assertJson([
                'success' => true,
                'order_status_updated' => false
            ]);

        $this->assertDatabaseHas('order_files', [
            'order_id' => $this->order->id,
            'uploader_id' => $this->writer->id,
            'file_category' => 'draft',
            'access_control' => 'writer_admin'
        ]);

        // Order status should not change for draft files
        $this->order->refresh();
        $this->assertEquals('in_progress', $this->order->order_status);
    }

    /** @test */
    public function writer_can_upload_complete_file()
    {
        $file = UploadedFile::fake()->create('complete.docx', 1024, 'application/vnd.openxmlformats-officedocument.wordprocessingml.document');

        $response = $this->actingAs($this->writer)
            ->postJson('/order-files/writer-upload', [
                'file' => $file,
                'order_id' => $this->order->id,
                'file_category' => 'complete',
                'description' => 'Final deliverable',
                'uploader_user_type' => 'writer'
            ]);

        $response->assertStatus(200)
            ->assertJson([
                'success' => true,
                'order_status_updated' => true
            ]);

        $this->assertDatabaseHas('order_files', [
            'order_id' => $this->order->id,
            'uploader_id' => $this->writer->id,
            'file_category' => 'complete',
            'access_control' => 'writer_admin'
        ]);

        // Order status should change to under_review
        $this->order->refresh();
        $this->assertEquals('under_review', $this->order->order_status);
        $this->assertNotNull($this->order->work_submitted_at);
    }

    /** @test */
    public function writer_cannot_upload_to_order_they_dont_own()
    {
        $otherOrder = Order::factory()->create([
            'user_id' => $this->client->id,
            'writer_id' => User::factory()->create(['user_type' => 'writer'])->id,
            'order_status' => 'in_progress'
        ]);

        $file = UploadedFile::fake()->create('test.docx', 1024, 'application/vnd.openxmlformats-officedocument.wordprocessingml.document');

        $response = $this->actingAs($this->writer)
            ->postJson('/order-files/writer-upload', [
                'file' => $file,
                'order_id' => $otherOrder->id,
                'file_category' => 'draft',
                'uploader_user_type' => 'writer'
            ]);

        $response->assertStatus(422);
    }

    /** @test */
    public function only_writers_can_use_writer_upload_endpoint()
    {
        $file = UploadedFile::fake()->create('test.docx', 1024, 'application/vnd.openxmlformats-officedocument.wordprocessingml.document');

        $response = $this->actingAs($this->client)
            ->postJson('/order-files/writer-upload', [
                'file' => $file,
                'order_id' => $this->order->id,
                'file_category' => 'draft',
                'uploader_user_type' => 'writer'
            ]);

        $response->assertStatus(422);
    }

    /** @test */
    public function complete_file_upload_updates_order_timeline_fields()
    {
        $file = UploadedFile::fake()->create('complete.docx', 1024, 'application/vnd.openxmlformats-officedocument.wordprocessingml.document');

        $this->actingAs($this->writer)
            ->postJson('/order-files/writer-upload', [
                'file' => $file,
                'order_id' => $this->order->id,
                'file_category' => 'complete',
                'description' => 'Final deliverable',
                'uploader_user_type' => 'writer'
            ]);

        $this->order->refresh();
        
        $this->assertEquals('under_review', $this->order->order_status);
        $this->assertNotNull($this->order->work_submitted_at);
        $this->assertNotNull($this->order->status_changed_at);
        $this->assertEquals($this->writer->id, $this->order->status_changed_by);
        $this->assertEquals('Writer submitted complete work', $this->order->status_change_reason);
    }
} 